/* 
 * handle_signals.c 
 *
 * Оголошує ігнорування сигналів SIGHUP, SIGINT, SIGQUIT, SIGPIPE 
 * і встановлює власний обробник для сигналів SIGTERM, SIGBUS, SIGFPE,
 * SIGILL, SIGSEGV, SIGSYS, SIGXCPU, SIGXFSZ, після чого входить у 
 * безкінечний цикл. По отриманню будь-якого з сигналів, які перехоплює,
 * виводить відповідне повідомлення і завершує роботу.
 * Демонструє підхід до організації асинхронної обробки сигналів
 * у професійній програмі. Застосовує функції sigaction(), sigprocmask(),
 * sigfillset(), sigemptyset().
 *
 */

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

size_t strlen_safe(const char *s);
void clean_up();
void sig_handler(int signum);
int handle_signals();


/* Безпечний в контексті асинхронної обробки сигналів варіант strlen().
   Повертає довжину рядка, на який указує s. */
size_t strlen_safe(const char *s)
{
        size_t n = 0;

        while (*s++ != '\0')
                n++;
        return n;
}

/* Функція, яка виконує дії, необхідні для коректного завершення
   роботи програми. Повинна бути безпечною в контексті асинхронної
   обробки сигналів. */
void clean_up()
{
        /* ... */
}

/* Обробник сигналів SIGTERM, SIGBUS, SIGFPE, SIGILL, SIGSEGV, SIGSYS,
   SIGXCPU, SIGXFSZ */
void sig_handler(int signum)
{
        int i;
        struct {
                int signum;
                char *msg;
        } sigmsg[] = {
                { SIGTERM, "Отримано сигнал завершення роботи." },
                { SIGBUS,  "Отримано сигнал про спробу доступу до"
                                        " невизначеної ділянки пам'яті." },
                { SIGFPE,  "Отримано сигнал про арифметичну помилку." },
                { SIGILL,  "Отримано сигнал про спробу виконати невірну"
                                                        " інструкцію." },
                { SIGSEGV, "Отримано сигнал про помилку звернення до"
                                                        " пам'яті." },
                { SIGSYS,  "Отримано сигнал про спробу виконати"
                                " некоректний системний виклик." },
                { SIGXCPU, "Отримано сигнал про вичерпання ліміту"
                                                " процесорного часу." },
                { SIGXFSZ, "Отримано сигнал про вичерпання ліміту на"
                                                        " розмір файлу." },
                { 0,  NULL }
        };

        /* Виконує дії, необхідні для коректного завершення
           роботи програми. */
        clean_up();

        /* Виводить (безпечним в контексті асинхронної обробки сигналів
           способом) повідомлення про причину завершення роботи. */
        for (i = 0; sigmsg[i].signum > 0; i++) {
                if (sigmsg[i].signum == signum ) {
                        write(STDERR_FILENO, sigmsg[i].msg,
                                        strlen_safe(sigmsg[i].msg));
                        write(STDERR_FILENO, "\n", 1);
                        break;
                }
        }

        /* Завершує роботу. Застосовується функція _exit(), оскільки
           exit() не є безпечною в контексті асинхронної обробки
           сигналів. */
        _exit(EXIT_FAILURE);
}

/* Корегує диспозицію сигналів.
   Повертає 0 у випадку успіху та інше значення у випадку помилки. */
int handle_signals()
{
        sigset_t set;
        struct sigaction act;

        /* Блокує всі сигнали. */
        if (sigfillset(&set) != 0) {
                fprintf(stderr, "Помилка при спробі проініціалізувати"
                                " набір сигналів: %s\n",
                                strerror(errno));
                return 1;
        }
        if (sigprocmask(SIG_SETMASK, &set, NULL) != 0) {
                fprintf(stderr, "Помилка при спробі заблокувати всі"
                                " сигнали: %s\n", strerror(errno));
                return 1;
        }

        memset(&act, 0, sizeof(act));
        /* На час роботи обробника всі сигнали будуть блокуватись. */
        if (sigfillset(&act.sa_mask) != 0) {
                fprintf(stderr, "Помилка при спробі проініціалізувати"
                                " набір сигналів: %s\n", strerror(errno));
                return 1;
        }
        /* Оголошує ігнорування сигналів SIGHUP, SIGINT, SIGQUIT
           (сигналів, які спричиняються діями користувача (від'єднанням
           терміналу, натисненням спеціальних клавіш) і не обробляються
           програмою), а також SIGPIPE (якщо він ігнорується, функція
           write() у відповідних ситуаціях буде повертати управління з
           відповідним кодом помилки, що, як правило, зручніше, ніж
           перехоплення сигналу). */
        act.sa_handler = SIG_IGN;
        if ((sigaction(SIGHUP, &act, NULL) != 0)
                || (sigaction(SIGINT, &act, NULL) != 0)
                || (sigaction(SIGQUIT, &act, NULL) != 0)
                || (sigaction(SIGPIPE, &act, NULL) != 0)) {
                fprintf(stderr, "Помилка при спробі оголосити ігнорування"
                        " сигналів SIGHUP, SIGINT, SIGQUIT, SIGPIPE: %s\n",
                                                        strerror(errno));
                return 1;
        }
        /* Визначає нову диспозицію для сигналів SIGTERM (сигналу
           завершення роботи), а також SIGBUS, SIGFPE, SIGILL, SIGSEGV,
           SIGSYS, SIGXCPU, SIGXFSZ (сигналів, що сповіщають про
           помилки). */
        act.sa_handler = sig_handler;
        if ((sigaction(SIGTERM, &act, NULL) != 0)
                || (sigaction(SIGBUS, &act, NULL) != 0)
                || (sigaction(SIGFPE, &act, NULL) != 0)
                || (sigaction(SIGILL, &act, NULL) != 0)
                || (sigaction(SIGSEGV, &act, NULL) != 0)
                || (sigaction(SIGSYS, &act, NULL) != 0)
                || (sigaction(SIGXCPU, &act, NULL) != 0)
                || (sigaction(SIGXFSZ, &act, NULL) != 0)) {
                fprintf(stderr, "Помилка при спробі встановити диспозицію"
                                " сигналу SIGTERM, SIGBUS, SIGFPE, SIGILL,"
                                " SIGSEGV, SIGSYS, SIGXCPU або SIGXFSZ:"
                                " %s.\n", strerror(errno));
                return 1;
        }

        /* Розблоковує всі сигнали. */
        if (sigemptyset(&set) != 0) {
                fprintf(stderr, "Помилка при спробі проініціалізувати"
                                " набір сигналів: %s.\n",
                                strerror(errno));
                return 1;
        }
        if (sigprocmask(SIG_SETMASK, &set, NULL) != 0) {
                fprintf(stderr, "Помилка при спробі розблокувати всі"
                                " сигнали: %s.\n", strerror(errno));
                return 1;
        }

        return 0;
}

int main()
{
        printf("Привіт, мій PID дорівнює %d.\n", (int) getpid());

        if (handle_signals() != 0)
                exit(EXIT_FAILURE);

        /* Далі повинна виконуватись якась корисна робота.  */
        while(1)
                pause();
}
